/****************************************************************************
FILE   : hello_procfile.c
SUBJECT: Skeleton for a module that uses the seq file API.
AUTHOR : (C) Copyright 2023 by Peter Chapin

This module creates a file in the /proc file system that uses the seq file API to display
"records" of information. This file is part of Vermont State University's CIS-4020 (Operating
Systems) class.

EXERCISE: Modify this module so that when the /proc file is read it outputs all primes less than
1,000,000. To do this you will need to compute the primes on demand. Don't just try increasing
the size of the primes[] array!

Please send comments or bug reports to

     Peter Chapin
     Computer Information Systems
     Vermont State University
     Williston, VT 05495
     peter.chapin@vermontstate.edu
****************************************************************************/

#include <linux/module.h>   /* Needed for any module.         */
#include <linux/kernel.h>   /* Needed for printk, etc.        */
#include <linux/init.h>     /* Needed for init/exit macros.   */
#include <linux/proc_fs.h>  /* Needed for /proc stuff.        */
#include <linux/seq_file.h> /* Needed for the seq_file stuff. */

/* This example outputs the prime numbers less than 100. */
#define PRIME_COUNT 25
int primes[PRIME_COUNT] = {
     2,  3,  5,  7, 11,
    13, 17, 19, 23, 29,
    31, 37, 41, 43, 47,
    53, 59, 61, 67, 71,
    73, 79, 83, 89, 97
};

/* Declare the sequence handling functions that we define (below). */
static void *hello_seq_start( struct seq_file *s, loff_t *pos );
static void *hello_seq_next ( struct seq_file *s, void *v, loff_t *pos );
static void  hello_seq_stop ( struct seq_file *s, void *v );
static int   hello_seq_show ( struct seq_file *s, void *v );

/* Gather our sequence handling functions into a seq_operations structure. */
static struct seq_operations hello_seq_ops = {
    .start = hello_seq_start,
    .next  = hello_seq_next,
    .stop  = hello_seq_stop,
    .show  = hello_seq_show
};


/*
 * hello_seq_start
 *
 * This function is called each time the application calls read(). It starts the process of
 * accumulating data to fill the application buffer. Return a pointer representing the current
 * item. Return NULL if there are no more items.
 */
static void *hello_seq_start( struct seq_file *s, loff_t *record_number )
{
    if( *record_number == PRIME_COUNT ) return NULL;
    return &primes[*record_number];
}


/*
 * hello_seq_next
 *
 * This function is called to compute the next record in the sequence given a pointer to the
 * current record (in bookmark). It returns a pointer to the new record (essentially, an updated
 * bookmark) and updates *record_number appropriately. Return NULL if there are no more items.
 */
static void *hello_seq_next( struct seq_file *s, void *bookmark, loff_t *record_number )
{
    (*record_number)++;

    /* In this simple example record_number is sufficient to find the next item. */
    if( *record_number == PRIME_COUNT ) return NULL;
    return &primes[*record_number];
}


/*
 * hello_seq_stop
 *
 * This function is called whenever an application buffer is filled (or when start or next
 * returns NULL. It can be used to undo any special preparations done in start (such as
 * deallocating auxillary memory that was allocated in start. In simple cases, you often do not
 * need to do anything in this function.
 */
static void hello_seq_stop( struct seq_file *s, void *bookmark )
{
    /* Nothing needed. */
}


/*
 * hello_seq_show
 *
 * This function is called after next to actually compute the output. It can use various seq_...
 * printing functions (such as seq_printf) to format the output. It returns 0 if successful or a
 * negative value if it fails.
 */
static int hello_seq_show( struct seq_file *s, void *bookmark )
{
    seq_printf( s, "%d\n", *(int *)bookmark );
    return 0;
}


/* Define the only file handling function we need. */
static int hello_open( struct inode *inode, struct file *file )
{
    return seq_open( file, &hello_seq_ops );
}


/* Use the seq_file file handling functions for some of the proc operations. */
static struct proc_ops hello_proc_ops = {
    .proc_open    = hello_open,
    .proc_read    = seq_read,  /* Look at source of seq_read to understand protocol. */
    .proc_lseek   = seq_lseek,
    .proc_release = seq_release
};


/*
 * hello_init
 */
static int __init hello_init( void )
{
    struct proc_dir_entry *entry;
    kuid_t user_id;
    kgid_t group_id;

    /* Create a new entry in the /proc file system. */
    if( ( entry = proc_create( "primes", S_IFREG | S_IRUGO, NULL, &hello_proc_ops ) ) == NULL ) {
      printk( KERN_ERR "hello_procfile NOT loaded. Error encountered creating /proc file.\n" );
      return -ENOMEM;
    }

    /* Fill in some fields of the proc_dir_entry */
    user_id.val = 0;
    group_id.val = 0;
    proc_set_user( entry, user_id, group_id );
    proc_set_size( entry, 0 ); /* A more appropriate size might be nice. */

    printk( KERN_INFO "hello_procfile loaded\n" );
    return 0;
}


/*
 * hello_exit
 */
static void __exit hello_exit( void )
{
    /* Locate the named proc entry and unregister it. */
    remove_proc_entry( "primes", NULL );
    printk( KERN_INFO "hello_procfile unloaded\n" );
}


/* Specify which functions do initialization and cleanup. */
module_init( hello_init );
module_exit( hello_exit );


/* Take care of some documentation tasks. */
MODULE_LICENSE( "GPL" );
MODULE_AUTHOR( "Peter Chapin <peter.chapin@vermontstate.edu>" );
MODULE_DESCRIPTION( "Module to display prime numbers." );

